/*
 * Copyright 2026 Danslav Slavenskoj, Lingenic LLC
 * License: CC0 1.0 - Public Domain
 * https://creativecommons.org/publicdomain/zero/1.0/
 * You may use this code for any purpose without attribution.
 *
 * Spec: https://hsvfile.com
 * Repo: https://github.com/LingenicLLC/HSV
 */

import java.util.*;

/**
 * HSV - Hierarchical Separated Values
 *
 * A text-based file format and streaming protocol using ASCII control characters.
 * Unlimited nesting (like JSON). No escaping required. Binary data supported.
 */
public class HSV {
    // Control characters
    public static final char SOH = '\u0001'; // Start of Header
    public static final char STX = '\u0002'; // Start of Text (data block)
    public static final char ETX = '\u0003'; // End of Text
    public static final char EOT = '\u0004'; // End of Transmission
    public static final char SO  = '\u000e'; // Shift Out (start nested)
    public static final char SI  = '\u000f'; // Shift In (end nested)
    public static final char DLE = '\u0010'; // Data Link Escape (binary mode)
    public static final char FS  = '\u001c'; // File/Record Separator
    public static final char GS  = '\u001d'; // Group/Array Separator
    public static final char RS  = '\u001e'; // Record/Property Separator
    public static final char US  = '\u001f'; // Unit/Key-Value Separator

    /**
     * Represents an HSV document with optional header and records
     */
    public static class Document {
        public Map<String, Object> header;
        public List<Map<String, Object>> records;

        public Document() {
            this.header = null;
            this.records = new ArrayList<>();
        }
    }

    /**
     * Parse HSV text into a Document
     */
    public static Document parse(String text) {
        // Extract binary sections first
        Map<String, String> binaries = new HashMap<>();
        text = extractBinarySections(text, binaries);

        Document doc = new Document();
        int i = 0;

        while (i < text.length()) {
            char c = text.charAt(i);

            if (c == SOH) {
                int stxPos = text.indexOf(STX, i + 1);
                if (stxPos == -1) {
                    i++;
                    continue;
                }

                String headerContent = text.substring(i + 1, stxPos);
                doc.header = parseObject(headerContent, binaries);

                int etxPos = text.indexOf(ETX, stxPos + 1);
                if (etxPos == -1) {
                    i = stxPos + 1;
                    continue;
                }

                String dataContent = text.substring(stxPos + 1, etxPos);
                for (String record : splitRespectingNesting(dataContent, FS)) {
                    Map<String, Object> obj = parseObject(record, binaries);
                    if (!obj.isEmpty()) {
                        doc.records.add(obj);
                    }
                }

                i = etxPos + 1;
            } else if (c == STX) {
                int etxPos = text.indexOf(ETX, i + 1);
                if (etxPos == -1) {
                    i++;
                    continue;
                }

                String dataContent = text.substring(i + 1, etxPos);
                for (String record : splitRespectingNesting(dataContent, FS)) {
                    Map<String, Object> obj = parseObject(record, binaries);
                    if (!obj.isEmpty()) {
                        doc.records.add(obj);
                    }
                }

                i = etxPos + 1;
            } else {
                i++;
            }
        }

        return doc;
    }

    private static String extractBinarySections(String text, Map<String, String> binaries) {
        StringBuilder result = new StringBuilder();
        int i = 0;
        int placeholderCount = 0;

        while (i < text.length()) {
            if (text.charAt(i) == DLE && i + 1 < text.length() && text.charAt(i + 1) == STX) {
                int j = i + 2;
                StringBuilder binaryData = new StringBuilder();

                while (j < text.length()) {
                    if (text.charAt(j) == DLE && j + 1 < text.length()) {
                        if (text.charAt(j + 1) == ETX) {
                            String placeholder = "\0BINARY" + placeholderCount + "\0";
                            binaries.put(placeholder, unescapeBinary(binaryData.toString()));
                            result.append(placeholder);
                            placeholderCount++;
                            i = j + 2;
                            break;
                        } else if (text.charAt(j + 1) == DLE) {
                            binaryData.append(DLE);
                            j += 2;
                            continue;
                        }
                    }
                    binaryData.append(text.charAt(j));
                    j++;
                }

                if (j >= text.length()) {
                    result.append(text.charAt(i));
                    i++;
                }
            } else {
                result.append(text.charAt(i));
                i++;
            }
        }

        return result.toString();
    }

    private static String unescapeBinary(String data) {
        StringBuilder result = new StringBuilder();
        int i = 0;

        while (i < data.length()) {
            if (data.charAt(i) == DLE && i + 1 < data.length() && data.charAt(i + 1) == DLE) {
                result.append(DLE);
                i += 2;
            } else {
                result.append(data.charAt(i));
                i++;
            }
        }

        return result.toString();
    }

    private static String restoreBinaries(String value, Map<String, String> binaries) {
        for (Map.Entry<String, String> entry : binaries.entrySet()) {
            value = value.replace(entry.getKey(), entry.getValue());
        }
        return value;
    }

    private static List<String> splitRespectingNesting(String text, char sep) {
        List<String> parts = new ArrayList<>();
        StringBuilder current = new StringBuilder();
        int depth = 0;

        for (int i = 0; i < text.length(); i++) {
            char c = text.charAt(i);
            if (c == SO) {
                depth++;
                current.append(c);
            } else if (c == SI) {
                depth--;
                current.append(c);
            } else if (c == sep && depth == 0) {
                parts.add(current.toString());
                current = new StringBuilder();
            } else {
                current.append(c);
            }
        }

        if (current.length() > 0 || !parts.isEmpty()) {
            parts.add(current.toString());
        }

        return parts;
    }

    private static Object parseValue(String value, Map<String, String> binaries) {
        value = restoreBinaries(value, binaries);

        // Check for nested structure (SO at start, SI at end)
        if (value.length() >= 2 && value.charAt(0) == SO && value.charAt(value.length() - 1) == SI) {
            String inner = value.substring(1, value.length() - 1);
            return parseObject(inner, binaries);
        }

        // Check for array
        if (value.indexOf(GS) != -1) {
            List<String> parts = splitRespectingNesting(value, GS);
            List<Object> arr = new ArrayList<>();
            for (String p : parts) {
                arr.add(parseValue(p, binaries));
            }
            return arr;
        }

        return value;
    }

    private static Map<String, Object> parseObject(String content, Map<String, String> binaries) {
        Map<String, Object> obj = new LinkedHashMap<>();

        List<String> props = splitRespectingNesting(content, RS);
        for (String prop : props) {
            List<String> parts = splitRespectingNesting(prop, US);
            if (parts.size() >= 2) {
                String k = parts.get(0);
                String v = String.join(String.valueOf(US), parts.subList(1, parts.size()));
                obj.put(k, parseValue(v, binaries));
            }
        }

        return obj;
    }

    // Self-test
    public static void main(String[] args) {
        System.out.println("==================================================");
        System.out.println("HSV Parser Tests (Java)");
        System.out.println("==================================================");

        int passed = 0;
        int failed = 0;

        // Test basic parsing
        try {
            Document result = parse(STX + "name" + US + "Alice" + RS + "age" + US + "30" + ETX);
            assert result.records.size() == 1;
            assert result.records.get(0).get("name").equals("Alice");
            assert result.records.get(0).get("age").equals("30");
            System.out.println("✓ Basic parsing");
            passed++;
        } catch (Exception e) {
            System.out.println("✗ Basic parsing: " + e.getMessage());
            failed++;
        }

        // Test multiple records
        try {
            Document result = parse(STX + "name" + US + "Alice" + FS + "name" + US + "Bob" + ETX);
            assert result.records.size() == 2;
            System.out.println("✓ Multiple records");
            passed++;
        } catch (Exception e) {
            System.out.println("✗ Multiple records: " + e.getMessage());
            failed++;
        }

        // Test arrays
        try {
            Document result = parse(STX + "tags" + US + "a" + GS + "b" + GS + "c" + ETX);
            @SuppressWarnings("unchecked")
            List<Object> tags = (List<Object>) result.records.get(0).get("tags");
            assert tags.size() == 3;
            System.out.println("✓ Array values");
            passed++;
        } catch (Exception e) {
            System.out.println("✗ Array values: " + e.getMessage());
            failed++;
        }

        // Test header
        try {
            Document result = parse(SOH + "hsv" + US + "1.0" + RS + "type" + US + "users" + STX + "name" + US + "Alice" + ETX);
            assert result.header != null;
            assert result.header.get("hsv").equals("1.0");
            assert result.header.get("type").equals("users");
            System.out.println("✓ SOH header");
            passed++;
        } catch (Exception e) {
            System.out.println("✗ SOH header: " + e.getMessage());
            failed++;
        }

        // Test nesting
        try {
            Document result = parse(STX + "user" + US + SO + "name" + US + "Alice" + RS + "email" + US + "a@b.com" + SI + ETX);
            @SuppressWarnings("unchecked")
            Map<String, Object> user = (Map<String, Object>) result.records.get(0).get("user");
            assert user.get("name").equals("Alice");
            assert user.get("email").equals("a@b.com");
            System.out.println("✓ SO/SI nesting");
            passed++;
        } catch (Exception e) {
            System.out.println("✗ SO/SI nesting: " + e.getMessage());
            failed++;
        }

        // Test deep nesting
        try {
            Document result = parse(STX + "data" + US + SO + "level1" + US + SO + "level2" + US + "deep" + SI + SI + ETX);
            @SuppressWarnings("unchecked")
            Map<String, Object> data = (Map<String, Object>) result.records.get(0).get("data");
            @SuppressWarnings("unchecked")
            Map<String, Object> level1 = (Map<String, Object>) data.get("level1");
            assert level1.get("level2").equals("deep");
            System.out.println("✓ Deep nesting");
            passed++;
        } catch (Exception e) {
            System.out.println("✗ Deep nesting: " + e.getMessage());
            failed++;
        }

        // Test newlines
        try {
            Document result = parse(STX + "text" + US + "line1\nline2\nline3" + ETX);
            assert result.records.get(0).get("text").equals("line1\nline2\nline3");
            System.out.println("✓ Newlines in values");
            passed++;
        } catch (Exception e) {
            System.out.println("✗ Newlines in values: " + e.getMessage());
            failed++;
        }

        // Test quotes
        try {
            Document result = parse(STX + "msg" + US + "He said \"hello\"" + ETX);
            assert result.records.get(0).get("msg").equals("He said \"hello\"");
            System.out.println("✓ Quotes (no escaping)");
            passed++;
        } catch (Exception e) {
            System.out.println("✗ Quotes: " + e.getMessage());
            failed++;
        }

        // Test mixed content
        try {
            Document result = parse("ignored" + STX + "name" + US + "Alice" + ETX + "also ignored");
            assert result.records.size() == 1;
            assert result.records.get(0).get("name").equals("Alice");
            System.out.println("✓ Mixed content");
            passed++;
        } catch (Exception e) {
            System.out.println("✗ Mixed content: " + e.getMessage());
            failed++;
        }

        // Test multiple blocks
        try {
            Document result = parse(STX + "a" + US + "1" + ETX + "junk" + STX + "b" + US + "2" + ETX);
            assert result.records.size() == 2;
            System.out.println("✓ Multiple blocks");
            passed++;
        } catch (Exception e) {
            System.out.println("✗ Multiple blocks: " + e.getMessage());
            failed++;
        }

        System.out.println("==================================================");
        System.out.println(passed + " passed, " + failed + " failed");
        System.out.println("==================================================");

        System.exit(failed > 0 ? 1 : 0);
    }
}
